*-------------------------------------------------------------------------------
* Description: Prepare DOL ETA data on state unemployment insurance programs.
* Author:      Brendan M. Price
*
* Note: See "Data Map" and "Reporting Instructions" at
*       https://oui.doleta.gov/unemploy/DataDownloads.asp.
*-------------------------------------------------------------------------------


* Process state identities
capture program drop process_states
program define process_states
	rename st state_abbrev
	quietly drop if inlist(state_abbrev, "ST", "TC")
	quietly merge m:1 state_abbrev using "$basepath/data/states.dta", assert(2 3) keep(3) keepusing(state_fips) nogenerate
    drop state_abbrev
end

* Process reporting date (for monthly or quarterly files)
capture program drop process_date
program define process_date
	syntax, freq(string)

	if "`freq'" == "monthly" {
		gen int tm = ym(year(date(rptdate, "MDY")), month(date(rptdate, "MDY")))
		format %tm tm
	}
	else if "`freq'" == "quarterly" {
		gen int tq = yq(year(date(rptdate, "MDY")), quarter(date(rptdate, "MDY")))
		format %tq tq
	}

	drop rptdate
end


*-------------------------------------------------------------------------------
* ETA 218: Benefit rights and experience
*
* Notes: Each initial claim is subjected to a monetary test to determine whether
* the claimant earned enough in UI-covered employment during the "base period"
* (typically, the first four of the most recent five completed quarters) to be
* eligible for UI. The ETA 218 files report the number of determinations and the
* subset of these for which the claimant had adequate earnings to be eligible.
* Some states impose additional requirements under the monetary test, so not
* everyone with sufficient wage credits is able to establish a "benefit year"
* and thus be potentially eligible for benefits. Claimants who establish a
* benefit year must still meet the non-monetary requirements reflected in the
* data reported on ETA 207.
*-------------------------------------------------------------------------------

* Load source data from the DOL ETA website
import delimited "https://oui.doleta.gov/unemploy/csv/ar218.csv", varnames(1) delimiters(",") clear

* Process state identities
process_states

* Process reporting date
process_date, freq(quarterly)

* Process UI determination variables
quietly destring c*, replace
rename c1 monetary_tests
rename c2 monetary_denials
rename c3 monetary_approvals
rename c4 benefit_years

* Process claim counts by potential benefit durations
rename c9  pbdsubt
rename c10 pbdlt10
rename c11 pbd1014
rename c12 pbd1519
rename c13 pbd2021
rename c14 pbd2223
rename c15 pbd2425
rename c23 pbd2627
rename c24 pbd2829
rename c25 pbd3031
rename c26 pbd3233
rename c27 pbdgt33

* Retain processed variables
keep state_fips tq monetary_* benefit_years pbd*
order state_fips tq monetary_* benefit_years pbd*

* Label variables
label variable tq                 "Year x quarter"
label variable monetary_tests     "Total monetary determinations (regular state UI programs)"
label variable monetary_denials   "Determinations with insufficient wage credits"
label variable monetary_approvals "Determinations with sufficient wage credits"
label variable benefit_years      "Number of claimants establishing benefit years"

foreach v of varlist pbd* {
	if substr("`v'", 4, 4) == "subt" {
		local w "all durations"
	}
	else if substr("`v'", 4, 4) == "lt10" {
		local w "<10 weeks"
	}
	else if substr("`v'", 4, 4) == "gt33" {
		local w ">33 weeks"
	}
	else {
		local w1 = substr("`v'", 4, 2)
		local w2 = substr("`v'", 6, 2)
		local w "`w1'-`w2' weeks"
	}

	label variable `v' "Potential benefit duration: `w'"
}

* Save to disk
compress
isid state_fips tq
sort state_fips tq
order state_fips tq monetary_* benefit_years pbd*
tsset state_fips tq, quarterly
save "$basepath/data/eta218.dta", replace


*-------------------------------------------------------------------------------
* ETA 539: Claims and Extended Benefits data
*
* Notes: This file contains the official tallies of initial and continued claims
* reflected in the weekly claims release. States' preliminary counts of initial
* claims for the latest reference week are reported in ETA 538, rather than ETA
* 539, but ETA doesn't release the data files associated with ETA 538. So, these
* data files reflect data that have already been revised.
*
* The time convention used in these files is that initial claims reported for
* week t are filed during, and pertain to, week t, whereas continued claims
* reported for week t are filed during t but pertain to week t - 1. The code
* below takes leads of the continuing claims variables to account for this.
*-------------------------------------------------------------------------------

* Load source data from the DOL ETA website
import delimited "https://oui.doleta.gov/unemploy/csv/ar539.csv", delimiters(",") clear

* Process state identities
process_states

* Process the reporting period (dated to the end of the week)
gen refdate = date(rptdate, "MDY")
format %td refdate

* Retain select variables
keep state_fips refdate c3 c8 c18
quietly destring c3 c8 c18, replace

* Claim counts
rename c3 initclaims
rename c8 contclaims
rename c18 covered_emp

* Continued claims reported in week t pertain to the preceding week
tsset state_fips refdate, daily delta(7)
replace contclaims = F.contclaims

* Label variables
label variable refdate     "End of reference week"
label variable initclaims  "Initial claims"
label variable contclaims  "Continued claims
label variable covered_emp "Covered employment"

* Save to disk
compress
isid state_fips refdate
sort state_fips refdate
order state_fips refdate initclaims contclaims covered_emp
tsset state_fips refdate, daily delta(7)
save "$basepath/data/eta539.dta", replace


*-------------------------------------------------------------------------------
* ETA 5159: Claims and payment activities, regular state programs
*
* Notes: Initial claims include both "new" initial claims, which represent
* the first initial claim filed by a given claimant for a given benefit year,
* and "additional" initial claims, which occur when a claimant who stopped
* filing continued claims for a UI program priot to exhausting benefits seeks to
* reopen an earlier claim.

* ETA data on "weeks compensated" include both "total" UI -- cases in which a
* claimant receives his/her full weekly benefit amount -- and "partial" or
* "part-total" UI, in which a claimant's benefits are prorated on account of
* below-usual but above-zero earnings. (Technically, "partial UI" refers to
* cases in which a claimant is working reduced hours at the same main employer,
* whereas "part-total" UI refers to cases where a claimant has lost a primary
* job but has positive earnings at one or more other, secondary jobs.)
*
* "First payments" reflect the first payment issued within the benefit year, so
* they arise only in the wake of "new" initial claims and not "additional" ones.
* "Final payments" reflect benefit exhaustions, not cases where a claimant exits
* UI prior to exhaustion (e.g., upon becoming reemployed).
*
* Lastly, this and other DOL ETA files distinguish intrastate claims from
* interstate claims, which can occur when a claimant files for UI a state other
* than the one in which he or she accrued UI eligibility and experienced a
* qualifying loss of work. The handling of interstate claims is quite confusing,
* but whether working with state-level or federal-level data, one must avoid
* double counting claims towards both the "agent state" (in which the paperwork
* was filed) and the "liable state" (which issues the payments).
*-------------------------------------------------------------------------------

* Load source data from the DOL ETA website
import delimited "https://oui.doleta.gov/unemploy/csv/ar5159.csv", delimiters(",") clear

* Process state identities
process_states

* Process reporting date
process_date, freq(monthly)

* Retain select variables
keep state_fips tm c1 c2 c3 c4 c5 c21 c22 c38 c39 c45 c46 c51 c56
quietly destring c1 c2 c3 c4 c5 c21 c22 c38 c39 c45 c46 c51 c56, replace

* Process initial claims, continued claims, weeks compensated, amounts compensated, first and last payments
rename c1  initclaims
rename c2  initclaims_intra_new
rename c3  initclaims_intra_addl
rename c4  initclaims_inter_fromagent
rename c5  initclaims_transitional
rename c21 contclaims_intra
rename c22 contclaims_inter_fromagent
rename c38 weeks_paid
rename c39 weeks_paid_nonpart
rename c45 amount_paid
rename c46 amount_paid_nonpart
rename c51 first_payments
rename c56 last_payments

* Compute weeks partially compensated
gen weeks_paid_part = max(0, weeks_paid - weeks_paid_nonpart)

* Compute weekly benefit amounts
gen amount_paid_part = amount_paid - amount_paid_nonpart
gen wba = amount_paid/weeks_paid
gen wba_part = amount_paid_part/weeks_paid_part
gen wba_nonpart = amount_paid_nonpart/weeks_paid_nonpart

* Sum intrastate and interstate continued claims
gen contclaims = contclaims_intra + contclaims_inter_fromagent
drop contclaims_intra contclaims_inter_fromagent

* Label variables
label variable tm                         "Year x month"
label variable initclaims                 "Initial claims"
label variable initclaims_intra_new       "Initial claims: new intrastate, excl. transitional"
label variable initclaims_intra_addl      "Initial claims: additional intrastate (i.e., reopenings)"
label variable initclaims_inter_fromagent "Initial claims: interstate, filed from agent state"
label variable initclaims_transitional    "Initial claims: transitional (i.e., straddling end of benefit year)"
label variable contclaims                 "Continued claims"
label variable weeks_paid                 "Weeks compensated"
label variable weeks_paid_part            "Weeks compensated, partial unemployment"
label variable weeks_paid_nonpart         "Weeks compensated, total unemployment"
label variable amount_paid                "Total amount paid"
label variable amount_paid_part           "Total amount paid, partial unemployment"
label variable amount_paid_nonpart        "Total amount paid, total unemployment"
label variable wba                        "Weekly benefit amount"
label variable wba_part                   "Weekly benefit amount, partial unemployment"
label variable wba_nonpart                "Weekly benefit amount, total unemployment"
label variable first_payments             "First payments"
label variable last_payments              "Final payments (i.e., exhaustions)"

* Save to disk
compress
isid state_fips tm
sort state_fips tm
order state_fips tm initclaims* contclaims weeks_paid weeks_paid_part weeks_paid_nonpart amount_paid amount_paid_part amount_paid_nonpart wba wba_part wba_nonpart first_payments last_payments
tsset state_fips tm, monthly
save "$basepath/data/eta5159.dta", replace


*-------------------------------------------------------------------------------
* ETA 9051: Time lapse of all continued weeks compensated except workshare
*
* Notes: These data report realized time lapses for payments issued in a given
* month; since they are backward-looking, the distribution of time delays
* reported for a given month is not exactly equal to the distribution of time
* delays ultimately observed among claims filed in that month. These timing
* issues are particularly severe at times when claim volumes are changing
* rapidly or in periods with extra-long delays (especially spring 2020), since
* the most severely delayed claims will still be outstanding at month end.
*-------------------------------------------------------------------------------

* Load source data from the DOL ETA website
import delimited "https://oui.doleta.gov/unemploy/csv/ar9051.csv", delimiters(",") clear

* Process state identities
process_states

* Process reporting date
process_date, freq(monthly)

* Intrastate UI claims by time lapse
rename c2  payments_intra_lapse_total
rename c10 payments_intra_lapse_00_07
rename c18 payments_intra_lapse_08_14
rename c26 payments_intra_lapse_15_21
rename c34 payments_intra_lapse_22_28
rename c42 payments_intra_lapse_29_35
rename c50 payments_intra_lapse_36_42
rename c58 payments_intra_lapse_43_49
rename c66 payments_intra_lapse_50_56
rename c74 payments_intra_lapse_57_63
rename c82 payments_intra_lapse_64_70
rename c90 payments_intra_lapse_71_up

* Interstate UI claims by time lapse
rename c6  payments_inter_lapse_total
rename c14 payments_inter_lapse_00_07
rename c22 payments_inter_lapse_08_14
rename c30 payments_inter_lapse_15_21
rename c38 payments_inter_lapse_22_28
rename c46 payments_inter_lapse_29_35
rename c54 payments_inter_lapse_36_42
rename c62 payments_inter_lapse_43_49
rename c70 payments_inter_lapse_50_56
rename c78 payments_inter_lapse_57_63
rename c86 payments_inter_lapse_64_70
rename c94 payments_inter_lapse_71_up

* Pool intrastate and interstate claims
quietly destring payments_intra_* payments_inter_*, replace
foreach g in "total" "00_07" "08_14" "15_21" "22_28" "29_35" "36_42" "43_49" "50_56" "57_63" "64_70" "71_up" {
	if "`g'" == "total" {
		local gsfx
	}
	else if "`g'" == "71_up" {
		local gsfx "_11plus_weeks"
	}
	else {
		local w = real(substr("`g'", 4, 2))/7
		if `w' < 10 local w 0`w'
		local gsfx "_`w'_weeks"
	}

	gen payments`gsfx' = payments_intra_lapse_`g' + payments_inter_lapse_`g'
	drop payments_*_lapse_`g'
}

* Retain select variables
keep state_fips tm payments payments_*_weeks
order state_fips tm payments payments_*_weeks

* Label variables
label variable tm "Year x month"
label variable payments "Total number of payments"
label variable payments_01_weeks "# of payments with time lapse to payment <= 7 days"
label variable payments_02_weeks "# of payments with time lapse to payment 8-14 days"
label variable payments_03_weeks "# of payments with time lapse to payment 15-21 days"
label variable payments_04_weeks "# of payments with time lapse to payment 22-28 days"
label variable payments_05_weeks "# of payments with time lapse to payment 29-35 days"
label variable payments_06_weeks "# of payments with time lapse to payment 36-42 days"
label variable payments_07_weeks "# of payments with time lapse to payment 43-49 days"
label variable payments_08_weeks "# of payments with time lapse to payment 50-56 days"
label variable payments_09_weeks "# of payments with time lapse to payment 57-63 days"
label variable payments_10_weeks "# of payments with time lapse to payment 64-70 days"
label variable payments_11plus_weeks "# of payments with time lapse to payment 71+ days"

* Save to disk
compress
isid state_fips tm
sort state_fips tm
order state_fips tm payments payments_*_weeks
tsset state_fips tm, monthly
save "$basepath/data/eta9051.dta", replace
