*-------------------------------------------------------------------------------
* Description: Plot the distribution of birth weight by child sex.
* Author:      Brendan M. Price
*
* Notes:
* 1. The spartan scheme's default colors are optimized for line series and are
*    arguably too bright when plotting solid regions. In this example, I scale
*    down each color a bit to obtain lighter shades.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load natality counts
use "$basepath/data/vital.dta", clear

* Bin births by birth weight (1st bin: <500)
keep if !missing(birth_weight)
replace birth_weight = max(birth_weight, 400)
replace birth_weight = min(birth_weight, 5000)
gen wtbin = floor(birth_weight/100) - 3
collapse (count) births = age (min) min_wt = birth_weight (max) max_wt = birth_weight, by(plural wtbin)

* Compute each bin's share of all births
bysort plural: egen totbirths = total(births)
gen share = 100 * births/totbirths

* Reflect singletons around the x-axis
replace share = -1 * share if plural == 0


* Plot the birth-weight distribution by plurality
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(bar share wtbin if plural == 0, horizontal lcolor(black) lwidth(vvthin) color(*0.8))
	(bar share wtbin if plural == 1, horizontal lcolor(black) lwidth(vvthin) color(*0.8)),
	xtitle("Percent of births")
	xlabel(-10(2)10)
	ytitle("Birth weight (grams)")
	yscale(range(0 48))
	ylabel(
		1 "{&le}500"
		7 "1000-1100"
		12 "1500-1600"
		17 "2000-2100"
		22 "2500-2600"
		27 "3000-3100"
		32 "3500-3600"
		37 "4000-4100"
		42 "4500-4600"
		47 "5000+")
	ytick(1(1)47)
	text(50 -5 "{it:Singleton births}")
	text(50 +5 "{it:Plural births}")
	plotregion(margin(t=5))
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(off);
#delimit cr

* Export the figure
graph export "$basepath/out/birth_weight.pdf", as(pdf) $serif replace
