*-------------------------------------------------------------------------------
* Description: Plot potential benefit duration in the German UI system.
* Author:      Brendan M. Price
*
* Notes:
* 1. This example illustrates how to create and plot data "from scratch", based
*    on institutional rules rather than a loadable dataset.
* 2. I use filled and hollow circles to illustrate discontinuous jumps in PBD.
*-------------------------------------------------------------------------------


* Prepare a dataset for plotting the benefit-accrual schedule
*-------------------------------------------------------------------------------

* Create a grid
clear
range months 0 84 4201

* Potential benefit duration in the absence of ceilings
gen byte pbd = (months >= 12) * ceil(2 * floor(months/4))

* Specify potential benefit duration by age
foreach c of numlist 12 18 22 26 32 {
	* Set PBD
	gen byte pbd`c' = min(pbd, `c')

	* Create gaps at the "step" points
	gen step`c' = pbd`c' if months >= 12 & months <= 2 * `c' & (months/4 == floor(months/4))
	replace pbd`c' = . if !missing(step`c')

	* Distinguish between bottom and top of each step
	rename step`c' step`c't
	gen step`c'b = pbd`c'[_n - 1] if !missing(step`c't)
}

gen stepb = step32b
gen stept = step32t
replace pbd = . if !missing(stept)


* Plot the benefit-accrual schedule by age and employment history
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(line pbd12 months if months <= 84, clcolor("scheme p1") lwidth(medthick) cmissing(n))
	(line pbd18 months if inrange(months, 28, 84), clcolor("scheme p2") lwidth(medthick) cmissing(n))
	(line pbd22 months if inrange(months, 40, 84), clcolor("scheme p3") lwidth(medthick) cmissing(n))
	(line pbd26 months if inrange(months, 48, 84), clcolor("scheme p4") lwidth(medthick) cmissing(n))
	(scatter step12b months, mlcolor("scheme p1") mfcolor(white) msymbol(circle))
	(scatter step12t months, mcolor("scheme p1") msymbol(circle))
	(scatter stepb months if inlist(months, 28, 32, 36), mlcolor("scheme p2") mfcolor(white) msymbol(circle))
	(scatter stept months if inlist(months, 28, 32, 36), mcolor("scheme p2") msymbol(circle))
	(scatter stepb months if inlist(months, 40, 44), mlcolor("scheme p3") mfcolor(white) msymbol(circle))
	(scatter stept months if inlist(months, 40, 44), mcolor("scheme p3") msymbol(circle))
	(scatter stepb months if inlist(months, 48, 52), mlcolor("scheme p4") mfcolor(white) msymbol(circle))
	(scatter stept months if inlist(months, 48, 52), mcolor("scheme p4") msymbol(circle)),
	xtitle("Months worked in the previous 7 years")
	xscale(range(0 84))
	xlabel(0(12)84, nogrid)
	ytitle("Potential duration (months)")
	yscale(range(0 28) titlegap(*10))
	ylabel(0 6 12 18 22 26, nogrid)
	text(13 54 "max if age <45")
	text(19 60 "...if 45{&minus}46")
	text(23 68 "...if 47{&minus}51")
	text(27 76 "...if 52{&minus}54")
	scale(*1.3)
	legend(off);
#delimit cr

* Export the figure
graph export "$basepath/out/german_pbd.pdf", as(pdf) $serif replace
