*-------------------------------------------------------------------------------
* Description: Plot flows to and from employment during the COVID-19 pandemic.
* Author:      Brendan M. Price
*
* Notes:
* 1. Stata's default x-axis labels and ticks can sometimes be confusing when
*    applied to calendar time. In this example, I place the tick marks on either
*    side of each data point, rather than placing a single tick beneath each
*    tick mark. This is tedious and not necessarily worth it.
* 2. I had to use a combination of a "twoway line" series and the "yline" option
*    to get a zero line that both is shown above the plotted series and extends
*    the full width of the plot region.
* 3. I deviated from the scheme's default colors (by using cranberry) because of
*    the natural association of red coloring with negative values.
* 4. In the legend, Stata defaulted to adding too much space between the second
*    and third legend columns. I adjusted margins, zeroed out the column gaps,
*    and added space padding in the first column to get the spacing right.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load unemployment counts by duration unemployed
quietly freduse LNS17100000 LNS17200000 LNS17400000 LNS17800000, clear

rename LNS17100000 flows_ue
rename LNS17200000 flows_ne
rename LNS17400000 flows_eu
rename LNS17800000 flows_en

* Extract date in year-month format
gen tm = ym(year(daten), month(daten))
format %tm tm

* Retain the pandemic period
keep if inrange(tm, tm(2019m1), tm(2022m11))

* Express flows in millions
foreach v of varlist flows_* {
	replace `v' = `v'/1000
}

* Compute net inflows
gen netflows = flows_ue + flows_ne - flows_eu - flows_en

* Create stacked bars, flipping the sign for outflows
gen zo = 0
gen p1 = flows_ue
gen p2 = flows_ue + flows_ne
gen n1 = -1 * flows_eu
gen n2 = -1 * (flows_eu + flows_en)

* Plot each data point as of mid-month
replace daten = mdy(month(daten), 15, year(daten))

* Specify x-axis labels and ticks
local xlabels
local xticks
local xmticks

forvalues y = 2019/2023 {
	* Plot year labels mid-year
	if `y' < 2023 {
		local xlabels `"`xlabels' `=td(30jun`y')' "`y'""'
	}

	* Plot major tick marks at year start/end
	local xticks `xticks' `=td(1jan`y')'

	* Plot minor tick marks at each month
	forvalues m = 1/12 {
		if `y' < 2023 {
			local xmticks `xmticks' `=mdy(`m', 1, `y')'
		}
	}
}


* Plot employment flows
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(rarea p1 zo daten, yaxis(1 2) color(ebblue*1.0))
	(rarea p2 p1 daten, yaxis(1 2) color(ebblue*0.5))
	(rarea n1 zo daten, yaxis(1 2) color(cranberry*1.0))
	(rarea n2 n1 daten, yaxis(1 2) color(cranberry*0.5))
	(line zo daten, color(gs10) lwidth(medthin))
	(connected netflows daten, yaxis(1 2) msymbol(circle) color(black) lwidth(medthick)),
	subtitle("Millions", position(1))
	xtitle("")
	xscale(range(`=td(15dec2018)' `=td(15jan2023)'))
	xlabel(`xlabels', noticks)
	xticks(`xticks', tposition(crossing) tlength(*5))
	xmticks(`xmticks', tposition(inside) tlength(vsmall))
	yscale(range(-30 15) axis(1))
	yscale(range(-30 15) axis(2))
	ylabel(-30(5)15, axis(1) tposition(inside))
	ylabel(-30(5)15, axis(2) tposition(inside))
	yline(0, lcolor(gs10))
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(rows(3) ring(0) position(5) region(color(none)) bmargin(b=8 r=0) colgap(zero)
		order(
			- "Net inflows:    " 6 "" -
			- "Inflows:" 1 "U {&rarr} E" 2 "N {&rarr} E"
			- "Outflows:" 3 "E {&rarr} U" 4 "E {&rarr} N"));
#delimit cr

* Export the figure
graph export "$basepath/out/lm_flows.pdf", as(pdf) $serif replace
