*-------------------------------------------------------------------------------
* Description: Plot cyclical sensitivity of manuf. emp. to industry labor share.
* Author:      Brendan M. Price
*
* Notes:
* 1. As in an earlier example, recession shading takes a lot of manual tweaking.
*    In the case of the coefplot command, one can use the option "addplot()" to
*    add recession bars. These would obscure a standard "yline()" option used to
*    show a zero line, so I use addplot() to create the zero line as well.
*-------------------------------------------------------------------------------


* Prepare the data and run a regression
*-------------------------------------------------------------------------------

* Load the NBER-CES Manufacturing Industry Database
use "https://data.nber.org/nberces/nberces5818v1/nberces5818v1_n2012.dta", clear

* Narrow the analysis period
keep if inrange(year, 1970, 2018)

* Compute average labor share over this period
bysort naics (year): egen lshare = mean(100 * pay/vadd)

* Compute log employment
gen ln_emp = 100 * ln(emp)

* Regress log employment on baseline labor share
quietly reg ln_emp i.naics i.year ib1990.year##c.lshare, vce(cluster naics)
estimates store m1


* Plot the coefficients of interest
*-------------------------------------------------------------------------------

* Label select years
local coeflabels
forvalues y = 1970/2018 {
	if mod(`y', 5) == 0 {
		local coeflabels `"`coeflabels' `y'.l = "`y'""'
	}
	else {
		local coeflabels `"`coeflabels' `y'.l = " ""'
	}
}

* Prepare recession shading
clear
set obs 50
gen rval = _n - 0.5
gen rlow = -2.2
gen rupp = 0.7

foreach r in "1973-1975" "1980-1982" "1990-1991" "2001-2001" "2007-2009" {
	local r1 = substr("`r'", 1, 4)
	local r2 = substr("`r'", 6, 4)

	local x1 = 0.5 + (`r1' - 1970)
	local x2 = 1.5 + (`r2' - 1970)

	#delimit ;
	local recessions `recessions'
		rarea rupp rlow rval if inrange(rval, `x1', `x2'),
		lwidth(none)
		fcolor(gs14)
		plotregion(margin(t=0 b=0))
		below
		||;
	#delimit cr
}

#delimit ;
coefplot (m1),
	keep(*.year#c*.lshare 1990b.year#co.lshare)
	recast(connected)
	rename(*.year#c.lshare = .l)
	coeflabels(`coeflabels')
	omitted
	baselevels
	vertical
	ytitle("Coef. on industry labor share (1990 = 0)")
	yscale(range(-2.2 0.7))
	ylabel(-2.0(0.5)0.5, format(%5.1f))
	xsize(8)
	ysize(4)
	scale(*1.3)
	addplot(
		`recessions'
		scatteri 0 0.5 0 49.5, recast(line) color(gs6) lpattern(solid) below)
	;
#delimit cr

* Export the figure
graph export "$basepath/out/manuf_reg.pdf", as(pdf) $serif replace
