*-------------------------------------------------------------------------------
* Description: Plot changes in manuf. emp. against industry labor share.
* Author:      Brendan M. Price
*
* Notes:
* 1. Given the many overlapping observations, transparent markers work better
*    than solid markers here.
* 2. Weighted markers help convey the importance of each observation.
* 3. The uppermost observations stretch beyond the uppermost y-axis label. It's
*    usually best to avoid that, but doing so here would mean either extending
*    the y-axis all the way to 200 (creating lots of white space) or labeling
*    the y-axis in 50 percent increments (creating too much clutter).
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load the NBER-CES Manufacturing Industry Database
use "https://data.nber.org/nberces/nberces5818v1/nberces5818v1_n2012.dta", clear

* Focus on the period before/after the surge in Chinese import competition
keep if inlist(year, 1990, 2007)

* Restructure the data
keep naics year emp pay vadd
reshape wide emp pay vadd, i(naics) j(year)

* Drop industries with missing baseline employment
drop if missing(emp1990)

* Tag textile and apparel industries (hit hard by import competition)
gen textile = inlist(floor(naics/1000), 313, 314, 315)

* Compute baseline labor share ($K per worker)
gen lshare = 100 * pay1990/vadd1990

* Drop an outlier with an invalid labor share
drop if lshare > 100

* Compute log change in employment
gen dlogemp = 100 * (ln(emp2007) - ln(emp1990))


* Plot industries' employment changes against their baseline labor shares
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(scatter dlogemp lshare if textile == 0 [aw = emp1990], mfcolor(%15) msymbol(circle) mlwidth(vthin))
	(scatter dlogemp lshare if textile == 1 [aw = emp1990], mfcolor(%15) msymbol(circle) mlwidth(medium)),
	xtitle("Baseline labor share (percent)")
	ytitle("100 x log change in employment")
	yline(0)
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(rows(2) ring(0) position(5) order(2 1) bmargin(b=5 r=5)
		label(2 "Textile and apparel")
		label(1 "Other industries"));
#delimit cr

* Export the figure
graph export "$basepath/out/manuf_scatter.pdf", as(pdf) $serif replace
