*-------------------------------------------------------------------------------
* Description: Plot the distribution of UI potential benefit duration.
* Author:      Brendan M. Price
*
* Notes:
* 1. Note that Stata's "graph bar" command is different from "twoway bar".
* 2. For reasons I don't understand, Stata defaulted to coloring the y-axis
*    black despite the fact that the scheme calls for the color gs10. I manually
*    applied gs10 to match the x-axis coloring and coloring used in other files.
*    Stata occasionally throws curveballs like this.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load state x quarter data
use "$basepath/data/eta218.dta", clear
keep state_fips tq pbd*

* Fix the analysis period
keep if inrange(tq, tq(1990q1), tq(2019q4))

* Aggregate across states
collapse (sum) pbd*, by(tq)

* Aggregate to decadal frequency
gen decade = 10 * floor(year(dofq(tq))/10)
collapse (sum) pbd*, by(decade)

* Combine bins <15 weeks
gen pbdlt15 = pbdlt10 + pbd1014
drop pbdlt10 pbd1014

* Combine bins >27 weeks
gen pbdgt27 = pbd2829 + pbd3031 + pbd3233 + pbdgt33
drop pbd2829 pbd3031 pbd3233 pbdgt33

* Reshape
rename pbdsubt totclaims
reshape long pbd, i(decade) j(bin_str) string

* Create a numeric coding
gen bin = .
replace bin = 1 if bin_str == "lt15"
replace bin = 2 if bin_str == "1519"
replace bin = 3 if bin_str == "2021"
replace bin = 4 if bin_str == "2223"
replace bin = 5 if bin_str == "2425"
replace bin = 6 if bin_str == "2627"
replace bin = 7 if bin_str == "gt27"

label define bin_lbl 1 "<15", replace
label define bin_lbl 2 "15-19", add
label define bin_lbl 3 "20-21", add
label define bin_lbl 4 "22-23", add
label define bin_lbl 5 "24-25", add
label define bin_lbl 6 "26-27", add
label define bin_lbl 7 ">27", add
label values bin bin_lbl

* Compute the share of claims falling within each bin
gen share = 100 * pbd/totclaims
keep decade bin share
order decade bin share
sort decade bin
reshape wide share, i(bin) j(decade)


* Create a bar graph (histogram) showing the pdf in each decade
*-------------------------------------------------------------------------------

#delimit ;
graph bar share1990 share2000 share2010,
	over(bin)
	bar(1, color(ebblue*0.4))
	bar(2, color(ebblue*0.6))
	bar(3, color(ebblue*1.0))
	b1title("Potential benefit duration (weeks)")
	ytitle("Percent of approved claims")
	yscale(lcolor(gs10))
	ylabel(#8)
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(rows(3) ring(0) position(11) bmargin(t=5 l=5)
		label(1 "1990s")
		label(2 "2000s")
		label(3 "2010s"));
#delimit cr

* Export the figure
graph export "$basepath/out/ui_pbd.pdf", as(pdf) $serif replace
