*-------------------------------------------------------------------------------
* Description: Plot the timeliness of benefit payments.
* Author:      Brendan M. Price
*
* Notes:
* 1. This example illustrates a (tedious) way to generate an alternative set of
*    date labels; in particular, the right panel uses ticks at monthly frequency
*    but labels years and quarters. I add an invisible second row of labels in
*    the left panel to get the same vertical spacing for both panels.
* 2. The legend series are rank-ordered as they appear from top to bottom.
* 3. The extra spacing in the left panel looks odd in isolation; this figure is
*    adapted from a piece I wrote using UI data starting in the early 1990s, but
*    the particular series shown here start in 1997.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load state x month data on the timeliness of payments
use "$basepath/data/eta9051.dta", clear

* Focus on periods with relatively complete state-level data
keep if tm >= tm(1997m6) & tm <= tm(2022m11)

* Interpolate and extrapolate missing data
fillin state_fips tm

foreach v of varlist payments payments_*_weeks {
	bysort state_fips (tm): replace `v' = `v'[_n - 1] if missing(`v')
}

* Count payments made within a given number of weeks
rename payments_*_weeks d*

gen within_1_weeks = d01
gen within_2_weeks = d01 + d02
gen within_4_weeks = d01 + d02 + d03 + d04
gen within_8_weeks = d01 + d02 + d03 + d04 + d05 + d06 + d07 + d08

* Aggregate to the national level
collapse (sum) payments within_*_weeks, by(tm)
tsset tm

* Compute the share of payments made within each horizon
foreach k of numlist 1 2 4 8 {
	gen share_within_`k'_weeks = 100 * within_`k'_weeks/payments
}


* Plot payment timeliness in the pre-pandemic period
*-------------------------------------------------------------------------------

* Construct a month counter aligned with years
gen ty = (year(dofm(tm)) - 0.5) + 1/12 * month(dofm(tm)) - 1/24

* Plot long time series at annual frequency
#delimit ;
twoway
	(line share_within_1_weeks ty if year(dofm(tm)) <= 2019, yaxis(1 2) lwidth(medium))
	(line share_within_2_weeks ty if year(dofm(tm)) <= 2019, yaxis(1 2) lwidth(medium))
	(line share_within_4_weeks ty if year(dofm(tm)) <= 2019, yaxis(1 2) lwidth(medium))
	(line share_within_8_weeks ty if year(dofm(tm)) <= 2019, yaxis(1 2) lwidth(medium)),
	title("{it:Pre-pandemic}", margin(b=0))
	subtitle("Percent", position(1) margin(b=1))
	xtitle("")
	xscale(range(1988 2022))
	xlabel(1994.5 `""1990s" " ""' 2004.5 "2000s" 2014.5 "2010s", noticks)
	xtick(1989.5(10)2019.5, tposition(inside))
	xmtick(1989.5(1)2019.5, tposition(inside))
	ytitle("", axis(1))
	ytitle("", axis(2))
	yscale(range(0 100) axis(1))
	yscale(range(0 100) axis(2))
	ylabel(none, axis(1))
	ylabel(0(10)100, axis(2) tposition(inside))
	ytick(0(10)100, axis(1) tposition(inside))
	ytick(0(10)100, axis(2) tposition(inside))
	plotregion(margin(b=0))
	legend(rows(4) order(4 3 2 1) ring(0) position(8) bmargin(large)
		label(1 "Within 1 week")
		label(2 "Within 2 weeks")
		label(3 "Within 4 weeks")
		label(4 "Within 8 weeks"));
#delimit cr

tempfile g1
graph save "`g1'"


* Plot payment timeliness from 2018 onward
*-------------------------------------------------------------------------------

* Construct a month counter that starts in January 2018
gen tt = tm - tm(2017m12)
sum tt
local maxtt = r(max)
local maxtt_adj = `maxtt' + 4

* Note the start of the pandemic
local pandemic = tm(2020m3) - tm(2017m12)

* Prepare date labels
local xlabels
foreach t of numlist 2(6)`maxtt_adj' {
	local y = 2018 + (ceil(`t'/12) - 1)

	if mod(`t' - 2, 12) == 0 {
		local xlabels `xlabels' `t' `""Q1" "`y'""'
	}
	else {
		local xlabels `xlabels' `t' "Q3"
	}
}

* Plot recent history
#delimit ;
twoway
	(line share_within_1_weeks tt if tm >= tm(2018m1), yaxis(1 2))
	(line share_within_2_weeks tt if tm >= tm(2018m1), yaxis(1 2))
	(line share_within_4_weeks tt if tm >= tm(2018m1), yaxis(1 2))
	(line share_within_8_weeks tt if tm >= tm(2018m1), yaxis(1 2)),
	title("{it:2018-present}", margin(b=0))
	subtitle("Percent", position(1) margin(b=1))
	xtitle("")
	xscale(range(0 `maxtt_adj'))
	xlabel(`xlabels', noticks)
	xtick(0.5(3)`=`maxtt_adj' - 0.5', tposition(inside))
	xmtick(0.5(1)`=`maxtt_adj' - 0.5', tposition(inside))
	xline(`pandemic', lcolor(gs10))
	ytitle("", axis(1))
	ytitle("", axis(2))
	yscale(range(0 100) axis(1))
	yscale(range(0 100) axis(2))
	ylabel(none, axis(1))
	ylabel(0(10)100, axis(2) tposition(inside))
	ytick(0(10)100, axis(1) tposition(inside))
	ytick(0(10)100, axis(2) tposition(inside))
	text(30 `=`pandemic' - 1' "{it:Start of}" "{it:pandemic}", size(medsmall) placement(east))
	plotregion(margin(b=0))
	legend(off);
#delimit cr

tempfile g2
graph save "`g2'"

* Combine panels
graph combine "`g1'" "`g2'", graphregion(margin(zero)) imargin(small) scale(*1.65) xsize(8) ysize(4)

* Export the figure
graph export "$basepath/out/ui_timeliness.pdf", as(pdf) $serif replace
