*-------------------------------------------------------------------------------
* Description: Plot average weekly benefit amounts among UI claimants by state.
* Author:      Brendan M. Price
*
* Notes:
* 1. Since the same concept is being plotted along both axes, a square figure
*    with a 45-degree line works well here.
* 2. Since I use horizontal labels along the y-axis, the left margin is wider
*    than the bottom margin. I used a larger "xsize" than "ysize" to compensate.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Prepare PCE deflator, indexed to 2019 dollars
freduse DPCERG3A086NBEA, clear
gen int year = real(substr(date, 1, 4))
rename DPCERG3A086NBEA pce
egen base_pce = total(pce * (year == 2019))
replace pce = pce/base_pce
keep year pce
tempfile pce
save `pce'

* Load state x quarter data
use "$basepath/data/eta5159.dta", clear
keep state_fips tm weeks_paid_nonpart amount_paid_nonpart

* Compare WBAs in select years at comparable points in the business cycle
gen year = year(dofm(tm))
keep if inlist(year, 2006, 2019)
collapse (sum) weeks_paid_nonpart amount_paid_nonpart, by(state_fips year)
gen wba = amount_paid_nonpart/weeks_paid_nonpart
keep state_fips year wba

* Deflate WBAs to 2019 dollars
merge m:1 year using `pce', assert(2 3) keep(3) nogenerate
replace wba = wba/pce
keep state_fips year wba

* Merge in state abbreviations
merge m:1 state_fips using "$basepath/data/states.dta", assert(2 3) keep(3) keepusing(state_abbrev) nogenerate

* Reshape data
reshape wide wba, i(state_fips) j(year)

* Label outliers
gen mlabel = state_abbrev if inlist(state_abbrev, "HI", "MA", "ND", "PR", "RI")
gen mlabvposition = .
replace mlabvposition = 10 if state_abbrev == "PR"
replace mlabvposition = 7 if state_abbrev == "HI"


* Plot average weekly benefit amounts in select years
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(scatter wba2019 wba2006, msize(small) mlabel(mlabel) mlabvposition(mlabvposition) mlabcolor(gs6))
	(function y = x, range(0 500) color(gs10)),
	xtitle("Average WBA in 2006")
	xscale(range(0 500))
	xlabel(0(100)500)
	ytitle("Average WBA in 2019")
	yscale(range(0 500))
	ylabel(0(100)500)
	xsize(4.5)
	ysize(4)
	scale(*1.3)
	graphregion(margin(r=2))
	legend(off);
#delimit cr

* Export the figure
graph export "$basepath/out/ui_wba.pdf", as(pdf) $serif replace
