*-------------------------------------------------------------------------------
* Description: Plot the overall and Black unemployment rates.
* Author:      Brendan M. Price
*
* Notes:
* 1. I often show just a single y-axis on the left, with the ytitle running
*    vertically along the left axis. Using two axes, as I do here, makes it
*    easier for readers to gauge recent movements. I commandeer the subtitle
*    option to illustrate another way of labeling the right axis.
* 2. I plot the recession bars before the data to ensure they're in the back.
* 3. I tell Stata to use the scheme's first two colors when plotting the two
*    series because inclusion of the recession bars disrupts the normal order.
* 4. I manually specify the axis ranges and labels to get the spacing I want.
* 5. I set the plotregion's bottom margin to zero to avoid ugly white space.
* 6. In the legend, I list the Black series first because it appears above the
*    overall unemployment rate. I drop the background to avoid blocking the
*    recession bars. I use the order() option to exclude the recession bars.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load unemployment rates for all (a) and Black (b) workers
freduse UNRATE LNS14000006, clear
rename UNRATE      urate_a
rename LNS14000006 urate_b

* Extract date in year-month format
gen tm = ym(year(daten), month(daten))
format %tm tm

* Retain periods with complete data
keep if !missing(urate_a, urate_b)


* Prepare recession shading
*-------------------------------------------------------------------------------

* Set the bar's y-axis values
gen rlow = 0
gen rupp = 26

* Center bars around months
gen tm_adj = tm - 0.5

* Specify recession dates
local r1 "1973m11 1975m3"
local r2 "1980m1  1980m7"
local r3 "1981m7  1982m11"
local r4 "1990m7  1991m3"
local r5 "2001m3  2001m11"
local r6 "2007m12 2009m6"
local r7 "2020m2  2020m4"

* Assemble a list of shaded regions
local shading
forvalues k = 1/7 {
	local k1 : word 1 of `r`k''
	local k2 : word 2 of `r`k''

	#delimit ;
	local shading `shading'
		(rarea rupp rlow tm_adj if inrange(tm, `=tm(`k1') - 0.1', `=tm(`k2') + 0.1'),
		yaxis(1 2)
		lpattern(solid)
		lwidth(none)
		fcolor(gs14)
		plotregion(margin(t=0 b=0)));
	#delimit cr
}


* Plot unemployment rates
*-------------------------------------------------------------------------------

#delimit ;
twoway
	`shading'
	(line urate_a tm, yaxis(1 2) color("scheme p1"))
	(line urate_b tm, yaxis(1 2) color("scheme p2")),
	subtitle("Percent of labor force", position(1))
	xscale(range(`=tm(1969m7)' `=tm(2025m6)'))
	xlabel(`=tm(1970m1)'(60)`=tm(2025m1)', format(%tmCCYY))
	yscale(range(0 26) axis(1))
	yscale(range(0 26) axis(2))
	ylabel(0(5)25, axis(1))
	ylabel(0(5)25, axis(2))
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(order(9 8) ring(0) pos(12) rows(2) region(color(none)) bmargin(t=5 l=5)
		label(8 "Overall")
		label(9 "Black or African American"));
#delimit cr

* Export the figure
graph export "$basepath/out/unemp_black.pdf", as(pdf) $serif replace
