*-------------------------------------------------------------------------------
* Description: Plot the number unemployed by reason for unemployment.
* Author:      Brendan M. Price
*
* Notes:
* 1. I usually prefer solid lines when plotting time series (despite the fact
*    that they'll be hard to distinguish if printed grayscale), but in this case
*    the multiple intersecting series warrant differentiated patterns as well as
*    colors. Dashed lines, especially short-dashed lines, receive less visual
*    emphasis by default, so I thicken them to equalize the emphasis.
* 2. Given the relatively narrow time period shown, including a tick mark for
*    every year works well (omitting them works well too). For visual balance,
*    I also included a tick mark at each unit on the y-axis.
* 3. Stata defaulted to including a pointless xtitle, which I therefore dropped.
* 4. Given the positioning of the plotted series, it was convenient to place the
*    legend within the plotregion. It was too close to the left axis by default,
*    so I widened its left margin.
*-------------------------------------------------------------------------------


* Prepare the data
*-------------------------------------------------------------------------------

* Load unemployment counts by reason for unemployment
quietly freduse UNEMPLOY LNS13023653 LNS13023705 LNS13026638 LNS13023569 LNS13023557 LNS13026637, clear

rename UNEMPLOY unemployed
rename LNS13023653 layoffs
rename LNS13023705 leavers
rename LNS13026638 permloss
rename LNS13023569 entrants
rename LNS13023557 reentrants
rename LNS13026637 tempended

* Extract date in year-month format
gen tm = ym(year(daten), month(daten))
format %tm tm

* Focus on the years after the CPS redesign but before the pandemic
keep if inrange(year(dofm(tm)), 1994, 2019)

* Pool job losers on layoff with those whose temporary job ended
replace layoffs = layoffs + tempended
drop tempended

* Pool new and returning entrants
replace entrants = entrants + reentrants
drop reentrants

* Express all series in millions
foreach v of varlist layoffs permloss leavers entrants {
	replace `v' = `v'/1000
}


* Plot unemployment by reason
*-------------------------------------------------------------------------------

#delimit ;
twoway
	(line layoffs tm, lpattern(solid))
	(line permloss tm, lpattern(longdash) lwidth(*1.1))
	(line leavers tm, lpattern(dash) lwidth(*1.2))
	(line entrants tm, lpattern(shortdash) lwidth(*1.3)),
	xtitle("")
	xscale(range(`=tm(1993m7)' `=tm(2020m6)'))
	xlabel(`=tm(1994m1)'(60)`=tm(2019m1)', format(%tmCCYY))
	xticks(`=tm(1994m1)'(12)`=tm(2020m1)')
	ytitle("Millions")
	ylabels(0(2)8)
	yticks(0(1)8)
	plotregion(margin(b=0))
	xsize(8)
	ysize(4)
	scale(*1.3)
	legend(rows(4) ring(0) position(11) region(color(none)) bmargin(l=5)
		label(1 "Temporary layoff or temp. job ended")
		label(2 "Permanent job loser")
		label(3 "Job leaver")
		label(4 "Labor force entrant"));
#delimit cr

* Export the figure
graph export "$basepath/out/unemp_reasons.pdf", as(pdf) $serif replace
