*-------------------------------------------------------------------------------
* Description: Show seasonal changes in the unemployment rate by sex/education.
* Author:      Brendan M. Price
*
* Notes:
* 1. This example uses "coefplot" to plot regression estimates and "grc1leg" to
*    combine two separate panels into a single graph. grc1leg is analogous to
*    "graph combine" but shows a single legend instead of one legend per panel.
* 2. The same y-axis title applies to both panels and it would be gratuitous to
*    show it twice. I could include a "ytitle" option for the left panel but not
*    for the right, but Stata would then compress the plotregion in the left
*    panel. Instead, I apply an "l1title" option to the combined graph. I also
*    added space padding within the l1title option to get the desired alignment.
* 3. grc1leg failed to respect the options "xsize" and "ysize", so I added a
*    "graph display" command to get the right sizing.
*-------------------------------------------------------------------------------


* Prepare the data and estimates
*-------------------------------------------------------------------------------

* Load n.s.a. unemployment rates by sex and educational attainment
freduse LNU04000049 LNU04027675 LNU04027676 LNU04027677 LNU04027678 LNU04000050 LNU04027679 LNU04027680 LNU04027681 LNU04027682, clear

rename LNU04000049 urate_men
rename LNU04027675 urate_men_lth
rename LNU04027676 urate_men_hsg
rename LNU04027677 urate_men_smc
rename LNU04027678 urate_men_clg
rename LNU04000050 urate_women
rename LNU04027679 urate_women_lth
rename LNU04027680 urate_women_hsg
rename LNU04027681 urate_women_smc
rename LNU04027682 urate_women_clg

* Extract date in year-month format
gen tm = ym(year(daten), month(daten))
format %tm tm
tsset tm

* Extract month
gen byte month = month(dofm(tm))

* Focus on the period after the CPS redesign but before the pandemic
keep if inrange(tm, tm(1994m1), tm(2019m12))

* Construct a linear spline in time (Dupraz et al. 2022; Price and Wasserman 2022)
#delimit ;
mkspline
	tmspline1 `=ym(2000, 04)'
	tmspline2 `=ym(2003, 10)'
	tmspline3 `=ym(2007, 01)'
	tmspline4 `=ym(2010, 06)'
	tmspline5 `=ym(2014, 10)'
	tmspline6 = tm,
displayknots;
#delimit cr

* Run seasonal regressions
foreach v of varlist urate_* {
	* Run specification
	quietly ivreg2 `v' ib10.month tmspline*, bw(25) robust small

	* Transform coefficients so that October effect is first
	local coeflist
	forvalues m = 0/12 {
		local mm = mod(`m' + 10, 12) + 12 * (mod(`m' + 10, 12) == 0)
		local coeflist "`coeflist' (coef`m': _b[`mm'.month])"
	}

	quietly nlcom `coeflist', post
	estimates store `v'
}


* Plot seasonal fluctuations in unemployment throughout the year
*-------------------------------------------------------------------------------

* Specify common options
local options "keep(coef*) recast(connected) offset(0) lpattern(solid) msize(medsmall) ciopts(lpattern(solid))"

* Create a separate panel for each sex
foreach g in "men" "women" {
	local G = proper("`g'")

	#delimit ;
	coefplot
		(urate_`g'_lth, `options' label("Less than a high school degree"))
		(urate_`g'_hsg, `options' label("High school graduate"))
		(urate_`g'_smc, `options' label("Some college"))
		(urate_`g'_clg, `options' label("Bachelors degree or higher")),
		coeflabels(
			coef0 = "Oct."
			coef1 = "Nov."
			coef2 = "Dec."
			coef3 = "Jan."
			coef4 = "Feb."
			coef5 = "Mar."
			coef6 = "Apr."
			coef7 = "May"
			coef8 = "June"
			coef9 = "July"
			coef10 = "Aug."
			coef11 = "Sept."
			coef12 = "Oct."
		)
		title("{it:`G'}")
		xtitle("")
		xlabel(, alternate)
		ytitle("")
		yscale(range(-1 4))
		ylabel(-1(1)4)
		yline(0)
		vertical
		legend(rows(2) size(*0.8));
	#delimit cr

	tempfile g`g'
	graph save "`g`g''"
}

* Combine panels
#delimit ;
grc1leg "`gmen'" "`gwomen'",
	l1title("      Percentage points (Oct. = 0)", size(*0.8))
	imargin(zero)
	scale(*1.6);
#delimit cr

* Export the figure
graph display, xsize(8)	ysize(4)
graph export "$basepath/out/unemp_seasonality_fullyear.pdf", as(pdf) $serif replace


* Plot changes in unemployment from October to January
*-------------------------------------------------------------------------------

* Specify common options
local options `"color("scheme p1") msymbol(circle) ciopts(lcolor("scheme p1") lpattern(solid))"'

#delimit ;
coefplot
	(urate_men_lth, keep(coef3) rename(coef3 = men1) `options')
	(urate_men_hsg, keep(coef3) rename(coef3 = men2) `options')
	(urate_men_smc, keep(coef3) rename(coef3 = men3) `options')
	(urate_men_clg, keep(coef3) rename(coef3 = men4) `options')
	(urate_women_lth, keep(coef3) rename(coef3 = women1) `options')
	(urate_women_hsg, keep(coef3) rename(coef3 = women2) `options')
	(urate_women_smc, keep(coef3) rename(coef3 = women3) `options')
	(urate_women_clg, keep(coef3) rename(coef3 = women4) `options'),
	order(men* . women*)
	title("")
	xtitle("Oct.-Jan. {&Delta} in unemp. rate (p.p.)")
	xscale(range(-0.1 3.6))
	xlabel(0(0.5)3.5, format(%5.1f))
	headings(
		men1 = "{it:Men}"
		women1 = "{it:Women}",
		gap(-0.5))
	coeflabels(
		men0 = "Overall"
		men1 = "Less than a high school degree"
		men2 = "High school graduate"
		men3 = "Some college"
		men4 = "Bachelor's degree or more"
		women0 = "Overall"
		women1 = "Less than a high school degree"
		women2 = "High school graduate"
		women3 = "Some college"
		women4 = "Bachelor's degree or more"
	)
	grid(none)
	xsize(6.5)
	ysize(4)
	scale(*1.4)
	legend(off);
#delimit cr

graph export "$basepath/out/unemp_seasonality_octjan.pdf", as(pdf) $serif replace
